﻿<?php
// 設定預設速度 ★
$defaultTypeSpeedMs = 45;  // 字元出現間隔（毫秒）
$defaultLinePauseMs = 650; // 每句打完停頓（毫秒）
?>

<!doctype html>
<html lang="zh-Hant">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Typewriter Dialogue</title>
  <style>
    body { font-family: system-ui, -apple-system, "Segoe UI", Arial, sans-serif; margin: 24px; }
    #typeArea {
      white-space: pre-wrap;   /* 保留換行並自動換行 */
      line-height: 1.8;
      font-size: 18px;
      border: 1px solid #ddd;
      padding: 16px;
      border-radius: 10px;
      min-height: 180px;
    }
    .hint { margin-top: 10px; font-size: 13px; color: #666; }
  </style>
</head>
<body>

  <h2>打字機對話展示</h2>
  <div id="typeArea">載入中…</div>
  <div class="hint">（逐句接續打字；速度可調︰無游標閃爍）</div>

  <script>
    // 指定的外部檔名 ★
    const DIALOGUE_URL = "dialogue-1-1.json";

    // PHP 預設值
    const FALLBACK_TYPE_SPEED_MS = <?php echo (int)$defaultTypeSpeedMs; ?>;
    const FALLBACK_LINE_PAUSE_MS = <?php echo (int)$defaultLinePauseMs; ?>;

    // 安全下限（避免 0 / 負數）
    const MIN_TYPE_SPEED_MS = 1;
    const MIN_LINE_PAUSE_MS = 0;

    const typeArea = document.getElementById("typeArea");

    function sleep(ms) {
      return new Promise(resolve => setTimeout(resolve, ms));
    }

    function toSafeNumber(v, fallback) {
      const n = Number(v);
      return Number.isFinite(n) ? n : fallback;
    }

    function clampMin(v, minValue) {
      return v < minValue ? minValue : v;
    }

    async function typeLine(targetEl, text, typeSpeedMs) {
      const s = (text === null || text === undefined) ? "" : String(text);
      for (let i = 0; i < s.length; i++) {
        targetEl.textContent += s[i];
        await sleep(typeSpeedMs);
      }
    }

    async function runDialogue(lines, typeSpeedMs, linePauseMs) {
      typeArea.textContent = ""; // 清空「載入中…」
      for (let i = 0; i < lines.length; i++) {
        await typeLine(typeArea, lines[i], typeSpeedMs);

        // 每句結尾加換行（最後一句不加）
        if (i !== lines.length - 1) {
          typeArea.textContent += "\n";
        }

        await sleep(linePauseMs);
      }
    }

    async function init() {
      try {
        const res = await fetch(DIALOGUE_URL, { cache: "no-store" });
        if (!res.ok) {
          throw new Error(`HTTP ${res.status} ${res.statusText || ""}`.trim());
        }

        let data;
        try {
          data = await res.json();
        } catch (jsonErr) {
          throw new Error("JSON 解析失敗（檔案不是有效 JSON 格式）");
        }

        const lines = Array.isArray(data?.lines) ? data.lines : [];

        let typeSpeedMs = toSafeNumber(data?.meta?.defaultTypeSpeedMs, FALLBACK_TYPE_SPEED_MS);
        let linePauseMs = toSafeNumber(data?.meta?.defaultLinePauseMs, FALLBACK_LINE_PAUSE_MS);

        typeSpeedMs = clampMin(typeSpeedMs, MIN_TYPE_SPEED_MS);
        linePauseMs = clampMin(linePauseMs, MIN_LINE_PAUSE_MS);

        if (!lines.length) {
          typeArea.textContent = "（dialogue-1-1.json 沒有提供 lines 陣列或內容為空）";
          return;
        }

        await runDialogue(lines, typeSpeedMs, linePauseMs);

      } catch (err) {
        typeArea.textContent =
          "讀取 dialogue-1-1.json 失敗：\n" +
          (err?.message ?? String(err)) +
          "\n\n請確認：\n- 檔案存在且同一層目錄\n- 內容為有效 JSON\n- 透過 http:// 或 https:// 開啟（不要用 file:// 直接開檔）";
      }
    }

    init();
  </script>

</body>
</html>
