﻿<?php
// 設定預設速度 ★
$defaultTypeSpeedMs = 50;  // 字元出現間隔（毫秒）
$defaultLinePauseMs = 650; // 每句打完停頓（毫秒）
?>
<!doctype html>
<html lang="zh-Hant">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <meta http-equiv="Content-Type" content="text/html; charset=utf-8">

  <title>Typewriter Dialogue</title>

  <style>
    body { font-family: system-ui, -apple-system, "Segoe UI", Arial, sans-serif; margin: 24px; }

    /* ★ 對話汽球外框（箭頭掛在外層，所以外層不能 overflow:hidden） */
    #typeArea{
      position: relative;
      border: 3px solid #ddd;
      padding: 16px;
      border-radius: 12px;
      background: #fff;
      box-sizing: border-box;

      overflow: visible; /* ★ 重要：讓箭頭可顯示 */
      width: fit-content;
      min-width: 360px;
      max-width: 90vw;
    }

    /* ★ 內層文字區：只顯示 2 行、超出隱藏（裁切文字，不裁切箭頭） */
    #typeText{
      position: relative;
      z-index: 2;         /* ★ 文字永遠在最上層 */
      white-space: pre-wrap;
      line-height: 1.8;
      font-size: 18px;

      height: 3.6em;      /* 2 行：1.8em * 2 */
      overflow: hidden;
    }

    /*
      ★ 長三角形箭頭：外框（灰）放在 ::before（下層）並用 rotate(-17deg) 讓尖端稍微往右指
    */

    #typeArea::before{
      content:"";
      position:absolute;
      z-index: 0;         /* ★ 外框在最底層 */
      right: 16px;
      bottom: -32px;
      width: 0;
      height: 0;

      border-style: solid;
      border-width: 36px 16px 0 16px; /* 高度36px、左右寬16px（可再加長） */
      border-color: #ddd transparent transparent transparent;

      transform: rotate(-17deg);
      transform-origin: top center;
    }

    /*
      ★ 長三角形箭頭：內填（白）放在 ::after（上層）

    */
    #typeArea::after{
      content:"";
      position:absolute;
      z-index: 1;         /* ★ 白色內填壓在外框上 */
      right: 18px;
      bottom: -28px;
      width: 0;
      height: 0;

      border-style: solid;
      border-width: 33px 13px 0 13px; /* 比外框小一圈 */
      border-color: #fff transparent transparent transparent;

      transform: rotate(-17deg);
      transform-origin: top center;
    }

    .hint { margin-top: 10px; font-size: 13px; color: #666; }
  </style>
</head>

<body bgcolor="#999999">

  <div id="typeArea"><div id="typeText">loading...</div></div>

  <script>
    // 指定的外部檔名 ★
    const DIALOGUE_URL = "dialogue-KMU.json";

    // PHP 預設值
    const FALLBACK_TYPE_SPEED_MS = <?php echo (int)$defaultTypeSpeedMs; ?>;
    const FALLBACK_LINE_PAUSE_MS = <?php echo (int)$defaultLinePauseMs; ?>;

    const typeText = document.getElementById("typeText");

    // ★ 顯示行數上限（只顯示 2 行）
    const MAX_VISIBLE_LINES = 2;

    function sleep(ms) {
      return new Promise(resolve => setTimeout(resolve, ms));
    }

    // ★ 只保留「上一行（已完成）」+「目前正在打的這一行」
    let historyLines = [];  // 最多保留 1 行（MAX_VISIBLE_LINES - 1）
    let currentLine = "";

    function renderTwoLines() {
      const above = historyLines.slice(-(MAX_VISIBLE_LINES - 1)); // 只取最後 1 行
      const linesToShow = [...above, currentLine];
      typeText.textContent = linesToShow.join("\n");
    }

    async function typeLine(text, typeSpeedMs) {
      currentLine = "";
      renderTwoLines();

      const s = (text === null || text === undefined) ? "" : String(text);
      for (let i = 0; i < s.length; i++) {
        currentLine += s[i];
        renderTwoLines();
        await sleep(typeSpeedMs);
      }
    }

    async function runDialogue(lines, typeSpeedMs, linePauseMs) {
      // 清空
      historyLines = [];
      currentLine = "";
      renderTwoLines();

      for (let i = 0; i < lines.length; i++) {
        await typeLine(lines[i], typeSpeedMs);

        // 最後一句打完就停在畫面上
        if (i === lines.length - 1) break;

        await sleep(linePauseMs);

        // ★ 把打完的句子變成「上一行」，並維持只保留 1 行
        historyLines.push(currentLine);
        if (historyLines.length > (MAX_VISIBLE_LINES - 1)) {
          historyLines.shift(); // 超過就把更早的擠掉（消失）
        }

        // 準備下一句
        currentLine = "";
        renderTwoLines();
      }
    }

    async function init() {
      try {
        const res = await fetch(DIALOGUE_URL, { cache: "no-store" });
        if (!res.ok) throw new Error("HTTP " + res.status);

        const data = await res.json();

        const lines = Array.isArray(data.lines) ? data.lines : [];
        const typeSpeedMs =
          Number.isFinite(data?.meta?.defaultTypeSpeedMs) ? data.meta.defaultTypeSpeedMs : FALLBACK_TYPE_SPEED_MS;

        const linePauseMs =
          Number.isFinite(data?.meta?.defaultLinePauseMs) ? data.meta.defaultLinePauseMs : FALLBACK_LINE_PAUSE_MS;

        if (!lines.length) {
          typeText.textContent = "（.json 沒有提供內容）";
          return;
        }

        await runDialogue(lines, typeSpeedMs, linePauseMs);
      } catch (err) {
        typeText.textContent = "讀取 .json 失敗：\n" + (err?.message ?? String(err));
      }
    }

    init();
  </script>

</body>
</html>
