﻿<?php

// 設定預設速度 ★
$defaultTypeSpeedMs = 45;  // 字元出現間隔（毫秒）
$defaultLinePauseMs = 650; // 每句打完停頓（毫秒）
?>

<!doctype html>
<html lang="zh-Hant">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Typewriter Dialogue</title>
  <style>
    body { font-family: system-ui, -apple-system, "Segoe UI", Arial, sans-serif; margin: 24px; }
    #typeArea {
      white-space: pre-wrap;   /* 保留換行並自動換行 */
      line-height: 1.8;
      font-size: 18px;
      border: 1px solid #ddd;
      padding: 16px;
      border-radius: 10px;

      /* ★ 固定只顯示 2 行（含 padding），超出的內容直接隱藏 */
      box-sizing: border-box;
      height: calc(3.6em + 32px); /* 2行高度(1.8em*2=3.6em) + 上下padding(16*2=32px) */
      overflow: hidden;
    }
    .hint { margin-top: 10px; font-size: 13px; color: #666; }
  </style>
</head>
<body>

  <h2>打字機對話展示</h2>
  <div id="typeArea">載入中…</div>
  <div class="hint">（逐句接續打字；速度可調︰無游標閃爍；只顯示 2 行）</div>

  <script>
    // 指定的外部檔名 ★
    const DIALOGUE_URL = "dialogue-1-1.json";

    // PHP 預設值
    const FALLBACK_TYPE_SPEED_MS = <?php echo (int)$defaultTypeSpeedMs; ?>;
    const FALLBACK_LINE_PAUSE_MS = <?php echo (int)$defaultLinePauseMs; ?>;

    const typeArea = document.getElementById("typeArea");

    // ★ 顯示行數上限（只顯示 2 行）
    const MAX_VISIBLE_LINES = 2;

    function sleep(ms) {
      return new Promise(resolve => setTimeout(resolve, ms));
    }

    // ★ 只保留「上一行（已完成）」+「目前正在打的這一行」
    let historyLines = [];  // 最多保留 1 行（MAX_VISIBLE_LINES - 1）
    let currentLine = "";

    function renderTwoLines() {
      const above = historyLines.slice(-(MAX_VISIBLE_LINES - 1)); // 只取最後 1 行
      const linesToShow = [...above, currentLine];
      typeArea.textContent = linesToShow.join("\n");
    }

    async function typeLine(text, typeSpeedMs) {
      currentLine = "";
      renderTwoLines();

      const s = (text === null || text === undefined) ? "" : String(text);
      for (let i = 0; i < s.length; i++) {
        currentLine += s[i];
        renderTwoLines();
        await sleep(typeSpeedMs);
      }
    }

    async function runDialogue(lines, typeSpeedMs, linePauseMs) {
      // 清空
      historyLines = [];
      currentLine = "";
      renderTwoLines();

      for (let i = 0; i < lines.length; i++) {
        await typeLine(lines[i], typeSpeedMs);

        // 最後一句打完就停在畫面上，不把它推進 history 變成空白
        if (i === lines.length - 1) break;

        await sleep(linePauseMs);

        // ★ 把打完的句子變成「上一行」，並維持只保留 1 行
        historyLines.push(currentLine);
        if (historyLines.length > (MAX_VISIBLE_LINES - 1)) {
          historyLines.shift(); // 超過就把更早的擠掉（消失）
        }

        // 準備下一句
        currentLine = "";
        renderTwoLines();
      }
    }

    async function init() {
      try {
        const res = await fetch(DIALOGUE_URL, { cache: "no-store" });
        if (!res.ok) throw new Error("HTTP " + res.status);

        const data = await res.json();

        const lines = Array.isArray(data.lines) ? data.lines : [];
        const typeSpeedMs =
          Number.isFinite(data?.meta?.defaultTypeSpeedMs) ? data.meta.defaultTypeSpeedMs : FALLBACK_TYPE_SPEED_MS;

        const linePauseMs =
          Number.isFinite(data?.meta?.defaultLinePauseMs) ? data.meta.defaultLinePauseMs : FALLBACK_LINE_PAUSE_MS;

        if (!lines.length) {
          typeArea.textContent = "（.json 沒有提供內容）";
          return;
        }

        await runDialogue(lines, typeSpeedMs, linePauseMs);
      } catch (err) {
        typeArea.textContent = "讀取 .json 失敗：\n" + (err?.message ?? String(err));
      }
    }

    init();
  </script>

</body>
</html>
