// typewriter.js
(() => {
  const cfg = window.TYPEWRITER_CONFIG || {};

  // 設定值（可從 index.php 覆蓋）
  const DIALOGUE_URL = cfg.dialogueUrl || "dialogue-KMU.json";
  const FALLBACK_TYPE_SPEED_MS = Number(cfg.fallbackTypeSpeedMs ?? 50);
  const FALLBACK_LINE_PAUSE_MS = Number(cfg.fallbackLinePauseMs ?? 650);
  const MAX_VISIBLE_LINES = Number(cfg.maxVisibleLines ?? 2);
  const TEXT_ELEMENT_ID = cfg.textElementId || "typeText";

  const typeText = document.getElementById(TEXT_ELEMENT_ID);
  if (!typeText) return;

  function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  // ★ 只保留「上一行（已完成）」+「目前正在打的這一行」
  let historyLines = [];  // 最多保留 1 行（MAX_VISIBLE_LINES - 1）
  let currentLine = "";

  function renderTwoLines() {
    const above = historyLines.slice(-(MAX_VISIBLE_LINES - 1)); // 只取最後 1 行
    const linesToShow = [...above, currentLine];
    typeText.textContent = linesToShow.join("\n");
  }

  async function typeLine(text, typeSpeedMs) {
    currentLine = "";
    renderTwoLines();

    const s = (text === null || text === undefined) ? "" : String(text);
    for (let i = 0; i < s.length; i++) {
      currentLine += s[i];
      renderTwoLines();
      await sleep(typeSpeedMs);
    }
  }

  async function runDialogue(lines, typeSpeedMs, linePauseMs) {
    // 清空
    historyLines = [];
    currentLine = "";
    renderTwoLines();

    for (let i = 0; i < lines.length; i++) {
      await typeLine(lines[i], typeSpeedMs);

      // 最後一句打完就停在畫面上
      if (i === lines.length - 1) break;

      await sleep(linePauseMs);

      // ★ 把打完的句子變成「上一行」，並維持只保留 1 行
      historyLines.push(currentLine);
      if (historyLines.length > (MAX_VISIBLE_LINES - 1)) {
        historyLines.shift(); // 超過就把更早的擠掉（消失）
      }

      // 準備下一句
      currentLine = "";
      renderTwoLines();
    }
  }

  async function init() {
    try {
      const res = await fetch(DIALOGUE_URL, { cache: "no-store" });
      if (!res.ok) throw new Error("HTTP " + res.status);

      const data = await res.json();

      const lines = Array.isArray(data.lines) ? data.lines : [];
      const typeSpeedMs =
        Number.isFinite(data?.meta?.defaultTypeSpeedMs) ? data.meta.defaultTypeSpeedMs : FALLBACK_TYPE_SPEED_MS;

      const linePauseMs =
        Number.isFinite(data?.meta?.defaultLinePauseMs) ? data.meta.defaultLinePauseMs : FALLBACK_LINE_PAUSE_MS;

      if (!lines.length) {
        typeText.textContent = "（.json 沒有提供內容）";
        return;
      }

      await runDialogue(lines, typeSpeedMs, linePauseMs);
    } catch (err) {
      typeText.textContent = "讀取 .json 失敗：\n" + (err?.message ?? String(err));
    }
  }

  init();
})();
